﻿#!/usr/bin/python
from struct import *
import time
import sys
import random
import io
import shutil
import os

IS_LINUX = True

##############################
# CRC16 function (for Modbus)
##############################
def crc16(s):
    CrcTable = (
    0X0000, 0XC0C1, 0XC181, 0X0140, 0XC301, 0X03C0, 0X0280, 0XC241,
    0XC601, 0X06C0, 0X0780, 0XC741, 0X0500, 0XC5C1, 0XC481, 0X0440,
    0XCC01, 0X0CC0, 0X0D80, 0XCD41, 0X0F00, 0XCFC1, 0XCE81, 0X0E40,
    0X0A00, 0XCAC1, 0XCB81, 0X0B40, 0XC901, 0X09C0, 0X0880, 0XC841,
    0XD801, 0X18C0, 0X1980, 0XD941, 0X1B00, 0XDBC1, 0XDA81, 0X1A40,
    0X1E00, 0XDEC1, 0XDF81, 0X1F40, 0XDD01, 0X1DC0, 0X1C80, 0XDC41,
    0X1400, 0XD4C1, 0XD581, 0X1540, 0XD701, 0X17C0, 0X1680, 0XD641,
    0XD201, 0X12C0, 0X1380, 0XD341, 0X1100, 0XD1C1, 0XD081, 0X1040,
    0XF001, 0X30C0, 0X3180, 0XF141, 0X3300, 0XF3C1, 0XF281, 0X3240,
    0X3600, 0XF6C1, 0XF781, 0X3740, 0XF501, 0X35C0, 0X3480, 0XF441,
    0X3C00, 0XFCC1, 0XFD81, 0X3D40, 0XFF01, 0X3FC0, 0X3E80, 0XFE41,
    0XFA01, 0X3AC0, 0X3B80, 0XFB41, 0X3900, 0XF9C1, 0XF881, 0X3840,
    0X2800, 0XE8C1, 0XE981, 0X2940, 0XEB01, 0X2BC0, 0X2A80, 0XEA41,
    0XEE01, 0X2EC0, 0X2F80, 0XEF41, 0X2D00, 0XEDC1, 0XEC81, 0X2C40,
    0XE401, 0X24C0, 0X2580, 0XE541, 0X2700, 0XE7C1, 0XE681, 0X2640,
    0X2200, 0XE2C1, 0XE381, 0X2340, 0XE101, 0X21C0, 0X2080, 0XE041,
    0XA001, 0X60C0, 0X6180, 0XA141, 0X6300, 0XA3C1, 0XA281, 0X6240,
    0X6600, 0XA6C1, 0XA781, 0X6740, 0XA501, 0X65C0, 0X6480, 0XA441,
    0X6C00, 0XACC1, 0XAD81, 0X6D40, 0XAF01, 0X6FC0, 0X6E80, 0XAE41,
    0XAA01, 0X6AC0, 0X6B80, 0XAB41, 0X6900, 0XA9C1, 0XA881, 0X6840,
    0X7800, 0XB8C1, 0XB981, 0X7940, 0XBB01, 0X7BC0, 0X7A80, 0XBA41,
    0XBE01, 0X7EC0, 0X7F80, 0XBF41, 0X7D00, 0XBDC1, 0XBC81, 0X7C40,
    0XB401, 0X74C0, 0X7580, 0XB541, 0X7700, 0XB7C1, 0XB681, 0X7640,
    0X7200, 0XB2C1, 0XB381, 0X7340, 0XB101, 0X71C0, 0X7080, 0XB041,
    0X5000, 0X90C1, 0X9181, 0X5140, 0X9301, 0X53C0, 0X5280, 0X9241,
    0X9601, 0X56C0, 0X5780, 0X9741, 0X5500, 0X95C1, 0X9481, 0X5440,
    0X9C01, 0X5CC0, 0X5D80, 0X9D41, 0X5F00, 0X9FC1, 0X9E81, 0X5E40,
    0X5A00, 0X9AC1, 0X9B81, 0X5B40, 0X9901, 0X59C0, 0X5880, 0X9841,
    0X8801, 0X48C0, 0X4980, 0X8941, 0X4B00, 0X8BC1, 0X8A81, 0X4A40,
    0X4E00, 0X8EC1, 0X8F81, 0X4F40, 0X8D01, 0X4DC0, 0X4C80, 0X8C41,
    0X4400, 0X84C1, 0X8581, 0X4540, 0X8701, 0X47C0, 0X4680, 0X8641,
    0X8201, 0X42C0, 0X4380, 0X8341, 0X4100, 0X81C1, 0X8081, 0X4040 )

    crcValue = 0xFFFF
    for ch in s:
        tmp=crcValue^(ord(ch))
        crcValue=(crcValue>> 8)^CrcTable[(tmp & 0xff)]
    return crcValue

############################
# boolean to int function
############################
def bool2int(boolArray):
    return sum(1<<i for i, b in enumerate(boolArray) if b)

def int2boolArray(num):
    boolArray = list()
    while(num):
        if(num&1):
            boolArray.append(1)
        else:
            boolArray.append(0)
        num = num>>1

    return boolArray


##############################
# RTU Class
##############################

class RtuClient:
    """RTU Client"""
    strDate = None
    logFileName = None
    lastLogFileName = None
    isChangeLogFileFlag = False # this Flag indicate Log File changed when a day pass
    lastLogTime = 0 #second

    # log parameter: "LOGFILE", "MAIL_UPLOAD", "FTP_UPLOAD"
    log_dir = "LOGFILE"
    log_ftpDir = "FTP_UPLOAD"
    log_mailDir = "MAIL_UPLOAD"

    # debug log output
    # messages from verbose to a little: DLOG_DEBUG > DLOG_INFO > DLOG_ERROR
    DLOG_NONE = 0
    DLOG_DEBUG = 1
    DLOG_INFO = 2
    DLOG_ERROR = 3
    DEBUG_LOG_MAX = 2000
    debugLogFile = 'rtuc_debug.log'
    debugLogFileBak = 'rtuc_debug.log.bak'
    debug_flag = 0
    def enable_debug(self, lvl=DLOG_DEBUG):
        self.debug_flag = lvl
    def disable_debug(self):
        self.debug_flag = 0

    def debugOut(self, msg):
        if(self.debug_flag):
            print msg

    def logOutput(self, lvl, msg):
        if(self.debug_flag):
            strTime = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(time.time()))
            # deal log message
            if(lvl==self.DLOG_DEBUG):
                lvlMsg = 'DEBUG'
            elif(lvl==self.DLOG_INFO):
                lvlMsg = 'INFO '
            elif(lvl==self.DLOG_ERROR):
                lvlMsg = 'ERROR'
            if(self.debug_flag <= lvl):
                debugString = lvlMsg + " [" + strTime + "] " + msg
                print debugString
                if(self.debugLogFile!="" and self.debugLogFile!=None):
                    # if debug log file size is too big, backup it and use new log file
                    # 2000 lines debug log message is about 122kB
                    try:
                        if(os.path.getsize(self.debugLogFile) > 256000):# Max. size = 256kB
                            shutil.move(self.debugLogFile, self.debugLogFileBak)
                    except Exception, e:
                        print("Error" + " [" + strTime + "] " + "backup log file: " + str(e))

                    # write log
                    try:
                        with open(self.debugLogFile, 'a') as debugLog:
                            debugLog.write(debugString+"\r\n")
                    except Exception, e:
                        print("Error" + " [" + strTime + "] " + "write debug log into the file")


    def logDebug(self, msg):
        self.logOutput(self.DLOG_DEBUG, msg)

    def logInfo(self, msg):
        self.logOutput(self.DLOG_INFO, msg)

    def logError(self, msg):
        self.logOutput(self.DLOG_ERROR, msg)


    def __init__(self, mbNumber=0):
        """init RTU spec. and data field
        mbNumber: >=1, <=11, first is LocalIO"""
        #- basic parameter
        self.serverIp = '192.168.27.40'
        self.serverPort = 10000

        if(IS_LINUX):
            self.mbComPort = '/dev/ttyS1'   #COM2 --> RS-485
        else:
            self.mbComPort = 'COM1'         #COM1 --> for PC
        self.baudrate = 115200
        self.databit = 8
        self.parity = 'N'
        self.stopbit = 1
        self.mbTimeout = 1000 # ms

        self.logTime = 1 # s
        self.logChangeTime = 60*60*1 # 1 hour (unit: sec.)

        #- rtu header field (init/data)
        self.stationId = 1
        self.deviceType = 3      # 0=G-4500, 1=GT-540, 2=RTU-130, 3=GRP-520
        self.dataTime = 5
        self.heartTime = 0

        self.isGps = 0
        self.gpsLen = 0
        self.isSdFull = 0

        #- rtu mb header
        self.mbNumber = mbNumber
        self.mbName = ['']*mbNumber
        self.mbId = [0]*mbNumber
        self.mbIp = ['192.168.27.67']*mbNumber #None: Modbus/TCP, or be RTU
        self.mbPort = [502]*mbNumber
        self.diN = [0]*mbNumber #Di channel number
        self.doN = [0]*mbNumber
        self.aiN = [0]*mbNumber
        self.aoN = [0]*mbNumber
        self.cntN = [0]*mbNumber
        self.diAddress = [0]*mbNumber
        self.doAddress = [0]*mbNumber
        self.aiAddress = [0]*mbNumber
        self.aoAddress = [0]*mbNumber
        self.cntAddress = [0]*mbNumber

        #- data field (initial packet)
        self.aiType = [0]*mbNumber
        self.aoType = [0]*mbNumber
        self.aiFormat = [0]*mbNumber
        self.aoFormat = [0]*mbNumber
        for i in range(mbNumber):
            self.aiType[i] = [0]*32
            self.aoType[i] = [0]*32

        #- data field (data packet)
        self.isValid = [0]*mbNumber #is MB data valid
        self.di = [0]*mbNumber
        self.do = [0]*mbNumber
        self.ai = [0]*mbNumber
        self.ao = [0]*mbNumber
        self.cnt = [0]*mbNumber
        for i in range(mbNumber):
            self.di[i] = [0]*32
            self.do[i] = [0]*32
            self.ai[i] = [0]*32
            self.ao[i] = [0]*32
            self.cnt[i] = [0]*32
        self.gps = ''

        #- rtu command list, must be send to modbus later
        self.rtuCmdList = list()
        #- additional data (external modify)
        self.netInterface = 0

    # more easy for reading than "asDict"
    def printAll(self):
        attrs = vars(self)
        for item in sorted(self.__dict__.items()):
            print item[0] + ' = ' + str(item[1])

    @property
    def asDict(self):
        return (self.__dict__)

    #== add
    def addMb(self, mbName, mbId, mbIp, mbPort, diN, doN, aiN, aoN, cntN
                , diAddress=0, doAddress=0, aiAddress=0, aoAddress=0, cntAddress=0
                , aiType=255, aiFormat=0, aoType=255, aoFormat=0):
        """
        mbName: device name
        mbId:   device id
        mbIp:   Modbus device ip
        mbport: Modbus device port
        """
        self.mbNumber+=1
        #- initial packet
        self.mbName.append(mbName)
        self.mbId.append(mbId)
        self.mbIp.append('192.168.27.67')
        self.mbPort.append(502)
        self.diN.append(diN)
        self.doN.append(doN)
        self.aiN.append(aiN)
        self.aoN.append(aoN)
        self.cntN.append(cntN)
        self.diAddress.append(diAddress)
        self.doAddress.append(doAddress)
        self.aiAddress.append(aiAddress)
        self.aoAddress.append(aoAddress)
        self.cntAddress.append(cntAddress)
        # ai/ao type & format
        if(type(aiType)=='list'):
            self.aiType.append(aiType)
        else:
            self.aiType.append([aiType]*aiN)
        self.aiFormat.append(aiFormat)
        if(type(aoType)=='list'):
            self.aoType.append(aoType)
        else:
            self.aoType.append([aoType]*aoN)
        self.aoFormat.append(aoFormat)
        #- data packet
        self.isValid.append(0)
        self.di.append([0]*diN)
        self.do.append([0]*doN)
        self.ai.append([0]*aiN)
        self.ao.append([0]*aoN)
        self.cnt.append([0]*cntN)

        return self.mbNumber


    def setRtuDataRandom(self, isAll=False):
        '''set RTU Data randomly for testing'''
        if(isAll == False):
            i = 1
            self.isValid[i] = 1
            needBytes = (self.diN[i]+7)/8
            for j in range(needBytes):
                self.di[i][j] = random.randint(0, 255)
            needBytes = (self.doN[i]+7)/8
            for j in range(needBytes):
                self.do[i][j] = random.randint(0, 255)
            for j in range(self.aiN[i]):
                self.ai[i][j] = random.randint(0, 2**16-1)
            for j in range(self.aoN[i]):
                self.ao[i][j] = random.randint(0, 2**16-1)
            for j in range(self.cntN[i]):
                self.cnt[i][j] = random.randint(0, 2**32-1)
        else:
            for i in range(self.mbNumber):
                self.isValid[i] = 1
                for j in range(self.diN[i]):
                    self.di[i][j] = random.randint(0, 1)
                for j in range(self.doN[i]):
                    self.do[i][j] = random.randint(0, 1)
                for j in range(self.aiN[i]):
                    self.ai[i][j] = random.randint(0, 2**16-1)
                for j in range(self.aoN[i]):
                    self.ao[i][j] = random.randint(0, 2**16-1)
                for j in range(self.cntN[i]):
                    self.cnt[i][j] = random.randint(0, 2**32-1)

    def makeInitPack(self):
        localtime = time.localtime(time.time())
        #- Header information
        header = '@'*16     #16s
        version = 4         # 1byte
        packType = 0        # 1byte, 0 = initial packet
        #netInterface = 0    # 1byte, hi-4bit:priority, low-4bit:interface
        dataLen = 1         # 2byte
        year = localtime.tm_year-2000           #B
        month = localtime.tm_mon           #B
        day = localtime.tm_mday             #B
        hour = localtime.tm_hour          #B
        minute = localtime.tm_min         #B
        second = localtime.tm_sec #30         #B
        reserve2 = 0        # 1

        #(16+1+1+2+1+1+2)=24byte don't put in
        data = ''
        data = data + pack('<BBB', year, month, day)
        data = data + pack('<BBB', hour, minute, second)
        data = data + pack('<LH', self.dataTime, self.heartTime)
        data = data + pack('<BHB', self.mbNumber, self.isGps, self.isSdFull)
        data = data + pack('<BBBBBBB', 0, 0, 0, 0, 0, 0, 0) #reserve, 7B

        #- data information
        for i in range(self.mbNumber):
            data = data + pack('<20sBB', self.mbName[i], self.mbId[i], reserve2)
            data = data + pack('<BBBBB', self.diN[i], self.doN[i], self.aiN[i], self.aoN[i], self.cntN[i])
            if(self.aiN[i]>0):
                for j in range(self.aiN[i]):
                    data = data + pack('<B', self.aiType[i][j])
                data = data + pack('<B', self.aiFormat[i])
            if(self.aoN[i]>0):
                for j in range(self.aoN[i]):
                    data = data + pack('<B', self.aoType[i][j])
                data = data + pack('<B', self.aoFormat[i])

        #- add 'Length' header, last 2-bytes are CRC16
        dataLen = len(data) + calcsize('<16sBBHBBH')+2
        data = pack('<16sBBHBBH', header, version, packType, \
                    self.stationId, self.deviceType, self.netInterface, dataLen) + data
        #- crc16
        crc = crc16(data)
        data = data + pack('<H', crc)

        return data


    def makeDataPack(self):
        localtime = time.localtime(time.time())
        #- Header information
        header = '@'*16     #16s
        version = 4         # 1byte
        packType = 1        # 1byte, 1 = data packet
        #netInterface = 0    # 1byte, hi-4bit:priority, low-4bit:interface
        dataLen = 1         # 2byte
        year = localtime.tm_year-2000           #B
        month = localtime.tm_mon           #B
        day = localtime.tm_mday             #B
        hour = localtime.tm_hour          #B
        minute = localtime.tm_min         #B
        second = localtime.tm_sec #30         #B
        reserve2 = 0        # 1

        data = ''
        data = data + pack('<BBB', year, month, day)
        data = data + pack('<BBB', hour, minute, second)
        data = data + pack('<LH', self.dataTime, self.heartTime)
        data = data + pack('<BHB', self.mbNumber, self.gpsLen, self.isSdFull)
        data = data + pack('<BBBBBBB', 0, 0, 0, 0, 0, 0, 0) #reserve, 7B

        #- data information
        for i in range(self.mbNumber):
            #MB header(Local)
            data = data + pack('<20sBB', self.mbName[i], self.mbId[i], self.isValid[i])
            data = data + pack('<BBBBB', self.diN[i], self.doN[i], self.aiN[i], self.aoN[i], self.cntN[i])
            #MB data
            needBytes = (self.diN[i]+7)/8
            for j in range(needBytes):
                data = data + pack('<B', bool2int(self.di[i][j*8:(j+1)*8]))
            needBytes = (self.doN[i]+7)/8
            for j in range(needBytes):
                data = data + pack('<B', bool2int(self.do[i][j*8:(j+1)*8]))
            for j in range(self.aiN[i]):
                data = data + pack('<H', self.ai[i][j])
            for j in range(self.aoN[i]):
                data = data + pack('<H', self.ao[i][j])
            for j in range(self.cntN[i]):
                data = data + pack('<L', self.cnt[i][j])

        #GPS data
        data = data + pack('<'+str(self.gpsLen)+'s', self.gps)

        #- add 'Length' header, last 2-bytes are CRC16
        dataLen = len(data) + calcsize('<16sBBHBBH')+2
        data = pack('<16sBBHBBH', header, version, packType, \
                    self.stationId, self.deviceType, self.netInterface, dataLen) + data

        #crc16
        crc = crc16(data)
        data = data + pack('<H', crc)

        return data

    def makeHeartPack(self):
        localtime = time.localtime(time.time())
        #- Header information
        header = '@'*16     #16s
        version = 4         # 1byte
        packType = 2        # 1byte, 2 = Heartbeat packet
        #netInterface = 0    # 1byte, hi-4bit:priority, low-4bit:interface
        dataLen = 1         # 2byte
        year = localtime.tm_year-2000           #B
        month = localtime.tm_mon           #B
        day = localtime.tm_mday             #B
        hour = localtime.tm_hour          #B
        minute = localtime.tm_min         #B
        second = localtime.tm_sec #30         #B
        reserve2 = 0        # 1

        #(16+1+1+2+1+1+2)=24byte don't put in
        data = ''
        data = data + pack('<BBB', year, month, day)
        data = data + pack('<BBB', hour, minute, second)
        data = data + pack('<LH', self.dataTime, self.heartTime)
        data = data + pack('<BHB', self.mbNumber, self.isGps, self.isSdFull)
        data = data + pack('<BBBBBBB', 0, 0, 0, 0, 0, 0, 0) #reserve, 7B

        #- add 'Length' header, last 2-bytes are CRC16
        dataLen = len(data) + calcsize('<16sBBHBBH')+2
        data = pack('<16sBBHBBH', header, version, packType, \
                    self.stationId, self.deviceType, self.netInterface, dataLen) + data
        #- crc16
        crc = crc16(data)
        data = data + pack('<H', crc)

        return data

    def findIndex(self, mbName, mbId):
        for index in range(self.mbNumber):
            if (self.mbName[index]==mbName and self.mbId[index]==mbId):
                return index
        return None

    def checkCommand(self, data):
        """check command from RTU Center
        just emplement Packet Type 0, 1 (don't support 2, 3)
        """
        headerLen = calcsize('<16sBBBHH')
        if( len(data)< headerLen):
            return None
        try:
            header, version, packType, reserve1, staID, dataLen = unpack('<16sBBBHH', data[:headerLen])
            if(len(data) == dataLen):
                cmdLen = dataLen - (headerLen+2) #2->CRC
                cmd = data[headerLen:-2]
                # packType=0: set DO/AO command, (5, 6, 15)=(1Do, 1Ao, multi-Do)
                if(packType==0):
                    #mbName, mbId, mbFc, mbCh, mbValue = unpack('<20sBBBH', cmd)
                    cmdFcLen = calcsize('<20sBB')
                    cmdFc = cmd[:cmdFcLen]
                    mbName, mbId, mbFc = unpack('<20sBB', cmdFc)
                    mbName = mbName.strip("\x00")
                    mbIndex = self.findIndex(mbName, mbId)
                    if(mbIndex == None):
                        return None
                    if (mbFc == 5):
                        mbCh, mbValue = unpack('<BH', cmd[cmdFcLen:])
                        self.logInfo("--> mbId=" + str(mbId) + ", mbFC=" + str(mbFc) + ", mbCh=" + str(mbCh) + ", mbValue=" + str(mbValue))
                        if(mbCh>=0 and mbCh<32):
                            if(mbValue==0xFF00):
                                self.rtuCmdList.append({'type':0, 'staID':staID, 'index':mbIndex, 'FC':mbFc, 'ch':mbCh, 'value':True})
                            else:
                                self.rtuCmdList.append({'type':0, 'staID':staID, 'index':mbIndex, 'FC':mbFc, 'ch':mbCh, 'value':False})
                    elif (mbFc == 6):
                        mbCh, mbValue = unpack('<BH', cmd[cmdFcLen:])
                        self.logInfo("--> mbId=" + str(mbId) + ", mbFC=" + str(mbFc) + ", mbCh=" + str(mbCh) + ", mbValue=" + str(mbValue) )
                        if(mbCh>=0 and mbCh<16):
                            self.rtuCmdList.append({'type':0, 'staID':staID, 'index':mbIndex, 'FC':mbFc, 'ch':mbCh, 'value':mbValue})
                    elif (mbFc == 15):
                        (byteCount,) = unpack('<B', cmd[cmdFcLen:cmdFcLen+1])
                        values = unpack('<'+'B'*byteCount, cmd[cmdFcLen+1:])
                        mbValue = 0
                        for i in range(byteCount):
                            mbValue += (values[i]<<((byteCount-i-1)*8))
                        self.logInfo( "--> mbId=" + str(mbId) + ", mbFC=" + str(mbFc) + ", byteCount=" + str(byteCount) + ", mbValue=" + str(mbValue) )
                        if(byteCount>=1 and byteCount<=4):
                            self.rtuCmdList.append({'type':0, 'staID':staID, 'index':mbIndex, 'FC':mbFc, 'ch':0, 'value':mbValue})
                    elif (mbFc == 100):
                        self.logInfo( "single Counter, unpack as: " )
                        mbCh, mbValue = unpack('<BL', cmd[cmdFcLen:])
                        self.logInfo ("--> mbId=" + str(mbId) + ", mbFC=" + str(mbFc) + ", mbCh=" + str(mbCh) + ", mbValue=" + str(mbValue) )
                        if(mbCh>=0 and mbCh<16):
                            self.rtuCmdList.append({'type':0, 'staID':staID, 'index':mbIndex, 'FC':mbFc, 'ch':mbCh, 'value':mbValue})

                # packType=1: Utility Command
                elif(packType==1):
                    self.logInfo( "Utility Command (not support): " )
                    #self.rtuCmdList.append({'type':1, 'staID':staID, 'cmd':cmd})
                    self.logInfo(cmd)
                elif(packType==2):#remove later, don't support this Type
                    # originally, this command only for M-7080B and GT-540
                    # 2 command for M-7080B:
                    # (1) 1, 16,  96, 2, 4, 0, 0, 0, 0
                    # (2) 1,  5, 132, 1, 1, 1
                    # but in GRP-520, counter function only support ET-7k
                    # 2014/08/25, because no mbName, can't find out mbIndex --> cannot support
                    self.logInfo( "Counter Command (not support): " )
                    #self.rtuCmdList.append({'type':1, 'cmd':cmd})
                    self.logInfo( str([aa for aa in cmd]) )

            else:
                return None

        except Exception, e:
            self.logError( " Error: " + str(e) )
            return None



'''
rtu_c = RtuClient()
print rtu_c.asDict
rtu_init_packet = rtu_c.makeInitPack()
print '[' + str(len(rtu_init_packet)) + ']: ' + rtu_init_packet
rtu_data_packet = rtu_c.makeDataPack()
print '[' + str(len(rtu_data_packet)) + ']: ' + rtu_data_packet
'''